<?php
/**
 * Plugin Name: BCV Exchange Rates
 * Plugin URI: https://smartsites.es/tasa-oficial-del-bcv/
 * Description: Plugin para mostrar los tipos de cambio del Banco Central de Venezuela (BCV) mediante shortcode. Actualización automática cada hora.
 * Version: 1.0.0
 * Author: SmartSites
 * Author URI: https://smartsites.es/
 * License: GPL v2 or later
 * Text Domain: bcv-exchange-rates
 */

if (!defined('ABSPATH')) {
    exit; // Salir si se accede directamente
}

// Cargar archivos necesarios
require_once plugin_dir_path(__FILE__) . 'config.php';
require_once plugin_dir_path(__FILE__) . 'includes/admin.php';
require_once plugin_dir_path(__FILE__) . 'includes/updater.php';
require_once plugin_dir_path(__FILE__) . 'includes/customizer.php';
require_once plugin_dir_path(__FILE__) . 'includes/customizer-ajax.php';
require_once plugin_dir_path(__FILE__) . 'includes/customizer-page.php';

class BCV_Exchange_Rates {
    
    private static $instance = null;
    private $transient_name = 'bcv_exchange_rates_data';
    private $cache_duration = HOUR_IN_SECONDS; // Actualizar cada hora
    
    public static function get_instance() {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    private function __construct() {
        add_shortcode('bcv_rates', array($this, 'render_shortcode'));
        add_action('wp_enqueue_scripts', array($this, 'enqueue_styles'));
        add_action('init', array($this, 'load_textdomain'));
        
        // Programar actualización automática
        if (!wp_next_scheduled('bcv_update_rates_event')) {
            wp_schedule_event(time(), 'hourly', 'bcv_update_rates_event');
        }
        add_action('bcv_update_rates_event', array($this, 'update_rates_cron'));
        
        // Limpiar cron al desactivar el plugin
        register_deactivation_hook(__FILE__, array($this, 'deactivate'));
    }
    
    public function load_textdomain() {
        load_plugin_textdomain('bcv-exchange-rates', false, dirname(plugin_basename(__FILE__)) . '/languages/');
    }
    
    public function enqueue_styles() {
        wp_enqueue_style(
            'bcv-exchange-rates-style',
            plugins_url('assets/css/style.css', __FILE__),
            array(),
            '1.0.0'
        );
    }
    
    public function render_shortcode($atts) {
        $atts = shortcode_atts(array(
            'currency' => '', // Moneda específica (EUR, USD, CNY, etc.)
            'format' => 'table', // 'table', 'list', 'cards'
            'show_date' => 'yes', // Mostrar fecha de actualización
            'show_flags' => 'yes', // Mostrar banderas
            'theme' => 'modern', // 'modern', 'classic', 'minimal'
        ), $atts, 'bcv_rates');
        
        $rates_data = $this->get_exchange_rates();
        
        if (empty($rates_data['rates'])) {
            return '<div class="bcv-error">' . __('No se pudieron obtener los tipos de cambio. Por favor, intente más tarde.', 'bcv-exchange-rates') . '</div>';
        }
        
        // Si se especifica una moneda específica
        if (!empty($atts['currency'])) {
            return $this->render_single_currency($rates_data, $atts);
        }
        
        // Renderizar según el formato solicitado
        switch ($atts['format']) {
            case 'list':
                return $this->render_list_format($rates_data, $atts);
            case 'cards':
                return $this->render_cards_format($rates_data, $atts);
            case 'table':
            default:
                return $this->render_table_format($rates_data, $atts);
        }
    }
    
    private function render_table_format($data, $atts) {
        $theme_class = 'bcv-theme-' . $atts['theme'];
        $html = '<div class="bcv-exchange-rates ' . $theme_class . '">';
        
        if ($atts['show_date'] === 'yes') {
            $html .= '<div class="bcv-last-updated">';
            $html .= '<span class="bcv-label">' . __('Actualizado:', 'bcv-exchange-rates') . '</span> ';
            $html .= '<span class="bcv-date">' . esc_html($data['last_updated']) . '</span>';
            $html .= '</div>';
        }
        
        $html .= '<table class="bcv-rates-table">';
        $html .= '<thead><tr>';
        $html .= '<th>' . __('Moneda', 'bcv-exchange-rates') . '</th>';
        $html .= '<th>' . __('Tasa BCV', 'bcv-exchange-rates') . '</th>';
        $html .= '</tr></thead><tbody>';
        
        foreach ($data['rates'] as $currency => $rate) {
            $flag_class = $this->get_flag_class($currency);
            $html .= '<tr>';
            $html .= '<td class="bcv-currency">';
            if ($atts['show_flags'] === 'yes') {
                $html .= '<span class="bcv-flag ' . $flag_class . '"></span>';
            }
            $html .= '<span class="bcv-code">' . esc_html($currency) . '</span>';
            $html .= '<span class="bcv-name">' . esc_html($this->get_currency_name($currency)) . '</span>';
            $html .= '</td>';
            $html .= '<td class="bcv-rate">' . esc_html(number_format($rate, 2, ',', '.')) . ' Bs</td>';
            $html .= '</tr>';
        }
        
        $html .= '</tbody></table>';
        $html .= '</div>';
        
        return $html;
    }
    
    private function render_cards_format($data, $atts) {
        $theme_class = 'bcv-theme-' . $atts['theme'];
        $html = '<div class="bcv-exchange-rates ' . $theme_class . '">';
        
        if ($atts['show_date'] === 'yes') {
            $html .= '<div class="bcv-last-updated">';
            $html .= '<span class="bcv-label">' . __('Actualizado:', 'bcv-exchange-rates') . '</span> ';
            $html .= '<span class="bcv-date">' . esc_html($data['last_updated']) . '</span>';
            $html .= '</div>';
        }
        
        $html .= '<div class="bcv-cards-container">';
        
        foreach ($data['rates'] as $currency => $rate) {
            $flag_class = $this->get_flag_class($currency);
            $html .= '<div class="bcv-card">';
            $html .= '<div class="bcv-card-header">';
            if ($atts['show_flags'] === 'yes') {
                $html .= '<span class="bcv-flag ' . $flag_class . '"></span>';
            }
            $html .= '<span class="bcv-code">' . esc_html($currency) . '</span>';
            $html .= '</div>';
            $html .= '<div class="bcv-card-body">';
            $html .= '<div class="bcv-rate">' . esc_html(number_format($rate, 2, ',', '.')) . '</div>';
            $html .= '<div class="bcv-currency-name">' . esc_html($this->get_currency_name($currency)) . '</div>';
            $html .= '</div>';
            $html .= '</div>';
        }
        
        $html .= '</div></div>';
        
        return $html;
    }
    
    private function render_list_format($data, $atts) {
        $theme_class = 'bcv-theme-' . $atts['theme'];
        $html = '<div class="bcv-exchange-rates ' . $theme_class . '">';
        
        if ($atts['show_date'] === 'yes') {
            $html .= '<div class="bcv-last-updated">';
            $html .= '<span class="bcv-label">' . __('Actualizado:', 'bcv-exchange-rates') . '</span> ';
            $html .= '<span class="bcv-date">' . esc_html($data['last_updated']) . '</span>';
            $html .= '</div>';
        }
        
        $html .= '<ul class="bcv-rates-list">';
        
        foreach ($data['rates'] as $currency => $rate) {
            $flag_class = $this->get_flag_class($currency);
            $html .= '<li class="bcv-rate-item">';
            if ($atts['show_flags'] === 'yes') {
                $html .= '<span class="bcv-flag ' . $flag_class . '"></span>';
            }
            $html .= '<span class="bcv-currency-info">';
            $html .= '<span class="bcv-code">' . esc_html($currency) . '</span>';
            $html .= '<span class="bcv-name">' . esc_html($this->get_currency_name($currency)) . '</span>';
            $html .= '</span>';
            $html .= '<span class="bcv-rate">' . esc_html(number_format($rate, 2, ',', '.')) . ' Bs</span>';
            $html .= '</li>';
        }
        
        $html .= '</ul></div>';
        
        return $html;
    }
    
    private function render_single_currency($data, $atts) {
        $currency = strtoupper($atts['currency']);
        
        if (!isset($data['rates'][$currency])) {
            return '<div class="bcv-error">' . sprintf(__('La moneda %s no está disponible.', 'bcv-exchange-rates'), $currency) . '</div>';
        }
        
        $rate = $data['rates'][$currency];
        $theme_class = 'bcv-theme-' . $atts['theme'];
        $flag_class = $this->get_flag_class($currency);
        
        $html = '<div class="bcv-exchange-rates bcv-single-currency ' . $theme_class . '">';
        $html .= '<div class="bcv-currency-display">';
        
        if ($atts['show_flags'] === 'yes') {
            $html .= '<span class="bcv-flag ' . $flag_class . '"></span>';
        }
        
        $html .= '<div class="bcv-currency-info">';
        $html .= '<span class="bcv-code">' . esc_html($currency) . '</span>';
        $html .= '<span class="bcv-name">' . esc_html($this->get_currency_name($currency)) . '</span>';
        $html .= '</div>';
        
        $html .= '<div class="bcv-rate-display">';
        $html .= '<span class="bcv-rate">' . esc_html(number_format($rate, 2, ',', '.')) . '</span>';
        $html .= '<span class="bcv-currency-bs">Bs</span>';
        $html .= '</div>';
        
        if ($atts['show_date'] === 'yes') {
            $html .= '<div class="bcv-last-updated">';
            $html .= '<span class="bcv-label">' . __('Actualizado:', 'bcv-exchange-rates') . '</span> ';
            $html .= '<span class="bcv-date">' . esc_html($data['last_updated']) . '</span>';
            $html .= '</div>';
        }
        
        $html .= '</div></div>';
        
        return $html;
    }
    
    private function get_exchange_rates() {
        // Intentar obtener datos de la caché
        $cached_data = get_transient($this->transient_name);
        
        if ($cached_data !== false) {
            return $cached_data;
        }
        
        // Si no hay caché, obtener datos actualizados
        return $this->fetch_exchange_rates();
    }
    
    private function fetch_exchange_rates() {
        // Cargar configuración
        $config = include plugin_dir_path(__FILE__) . 'config.php';
        
        // Usar el actualizador mejorado
        $updater = BCV_Updater::get_instance();
        $data = $updater->fetch_rates();
        
        // Guardar en caché
        set_transient($this->transient_name, $data, $this->cache_duration);
        
        // Guardar respaldo si está habilitado
        if ($config['cache']['backup_duration']) {
            set_transient('bcv_exchange_rates_backup', $data, $config['cache']['backup_duration']);
        }
        
        return $data;
    }
    
    public function update_rates_cron() {
        $this->fetch_exchange_rates();
    }
    
    private function get_currency_name($code) {
        $names = array(
            'USD' => 'Dólar Americano',
            'EUR' => 'Euro',
            'CNY' => 'Yuan Chino',
            'TRY' => 'Lira Turca',
            'RUB' => 'Rublo Ruso',
        );
        
        return isset($names[$code]) ? $names[$code] : $code;
    }
    
    private function get_flag_class($code) {
        $flags = array(
            'USD' => 'flag-us',
            'EUR' => 'flag-eu',
            'CNY' => 'flag-cn',
            'TRY' => 'flag-tr',
            'RUB' => 'flag-ru',
        );
        
        return isset($flags[$code]) ? $flags[$code] : 'flag-default';
    }
    
    public function deactivate() {
        wp_clear_scheduled_hook('bcv_update_rates_event');
        delete_transient($this->transient_name);
    }
}

// Inicializar el plugin
BCV_Exchange_Rates::get_instance();

// Función auxiliar para uso en plantillas
function bcv_get_exchange_rates($currency = '', $format = 'table', $args = array()) {
    $plugin = BCV_Exchange_Rates::get_instance();
    
    $atts = array_merge($args, array(
        'currency' => $currency,
        'format' => $format,
    ));
    
    return $plugin->render_shortcode($atts);
}