<?php
/**
 * BCV Exchange Rates Admin Functions
 * Panel de administración y configuración
 */

if (!defined('ABSPATH')) {
    exit;
}

class BCV_Admin {
    
    private static $instance = null;
    
    public static function get_instance() {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    private function __construct() {
        add_action('admin_menu', array($this, 'add_admin_menu'));
        add_action('admin_init', array($this, 'register_settings'));
        add_action('admin_enqueue_scripts', array($this, 'enqueue_admin_scripts'));
        add_action('wp_dashboard_setup', array($this, 'add_dashboard_widget'));
        
        // Notificaciones de administración
        add_action('admin_notices', array($this, 'admin_notices'));
    }
    
    /**
     * Agregar menú de administración
     */
    public function add_admin_menu() {
        add_menu_page(
            __('BCV Exchange Rates', 'bcv-exchange-rates'),
            __('BCV Rates', 'bcv-exchange-rates'),
            'manage_options',
            'bcv-exchange-rates',
            array($this, 'admin_page'),
            'dashicons-chart-line',
            30
        );
        
        add_submenu_page(
            'bcv-exchange-rates',
            __('Configuración', 'bcv-exchange-rates'),
            __('Configuración', 'bcv-exchange-rates'),
            'manage_options',
            'bcv-settings',
            array($this, 'settings_page')
        );
        
        add_submenu_page(
            'bcv-exchange-rates',
            __('Herramientas', 'bcv-exchange-rates'),
            __('Herramientas', 'bcv-exchange-rates'),
            'manage_options',
            'bcv-tools',
            array($this, 'tools_page')
        );
        
        add_submenu_page(
            'bcv-exchange-rates',
            __('Personalizar Apariencia', 'bcv-exchange-rates'),
            __('Personalizar', 'bcv-exchange-rates'),
            'manage_options',
            'bcv-customizer',
            array($this, 'customizer_page')
        );
    }
    
    /**
     * Página principal de administración
     */
    public function admin_page() {
        if (!current_user_can('manage_options')) {
            wp_die(__('No tienes permisos suficientes para acceder a esta página.'));
        }
        
        $status = bcv_get_service_status();
        $last_update = BCV_Updater::get_instance()->get_last_update_time();
        ?>
        <div class="wrap">
            <h1><?php echo esc_html(get_admin_page_title()); ?></h1>
            
            <div class="bcv-admin-container">
                <div class="bcv-admin-grid">
                    <!-- Estado del Servicio -->
                    <div class="bcv-card">
                        <h2><?php _e('Estado del Servicio', 'bcv-exchange-rates'); ?></h2>
                        <div class="bcv-status-indicator <?php echo esc_attr($status['status']); ?>">
                            <?php echo esc_html($status['status'] === 'success' ? 'Funcionando' : 'Error'); ?>
                        </div>
                        <p><strong><?php _e('Última actualización:', 'bcv-exchange-rates'); ?></strong> <?php echo esc_html($last_update); ?></p>
                        <p><strong><?php _e('Próxima actualización:', 'bcv-exchange-rates'); ?></strong> <?php echo esc_html($status['next_update']); ?></p>
                        <?php if ($status['status'] === 'error' && !empty($status['message'])): ?>
                            <p class="bcv-error-message"><?php echo esc_html($status['message']); ?></p>
                        <?php endif; ?>
                    </div>
                    
                    <!-- Tipos de Cambio Actuales -->
                    <div class="bcv-card">
                        <h2><?php _e('Tipos de Cambio Actuales', 'bcv-exchange-rates'); ?></h2>
                        <?php
                        $data = get_transient('bcv_exchange_rates_data');
                        if ($data && !empty($data['rates'])):
                            ?>
                            <table class="bcv-rates-table">
                                <thead>
                                    <tr>
                                        <th><?php _e('Moneda', 'bcv-exchange-rates'); ?></th>
                                        <th><?php _e('Tasa BCV', 'bcv-exchange-rates'); ?></th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php foreach ($data['rates'] as $currency => $rate): ?>
                                        <tr>
                                            <td><?php echo esc_html($currency); ?></td>
                                            <td><?php echo number_format($rate, 2, ',', '.'); ?> Bs</td>
                                        </tr>
                                    <?php endforeach; ?>
                                </tbody>
                            </table>
                        <?php else: ?>
                            <p><?php _e('No hay datos disponibles.', 'bcv-exchange-rates'); ?></p>
                        <?php endif; ?>
                    </div>
                    
                    <!-- Acciones Rápidas -->
                    <div class="bcv-card">
                        <h2><?php _e('Acciones Rápidas', 'bcv-exchange-rates'); ?></h2>
                        <button type="button" class="button button-primary" id="bcv-force-update">
                            <?php _e('Forzar Actualización', 'bcv-exchange-rates'); ?>
                        </button>
                        <button type="button" class="button" id="bcv-clear-cache">
                            <?php _e('Limpiar Caché', 'bcv-exchange-rates'); ?>
                        </button>
                        <div id="bcv-update-result"></div>
                    </div>
                    
                    <!-- Información del Plugin -->
                    <div class="bcv-card">
                        <h2><?php _e('Información', 'bcv-exchange-rates'); ?></h2>
                        <p><strong><?php _e('Versión:', 'bcv-exchange-rates'); ?></strong> 1.0.0</p>
                        <p><strong><?php _e('Autor:', 'bcv-exchange-rates'); ?></strong> Alain Milian Marin</p>
                        <p><strong><?php _e('Shortcode:', 'bcv-exchange-rates'); ?></strong> <code>[bcv_rates]</code></p>
                        <p><strong><?php _e('Shortcode Cards:', 'bcv-exchange-rates'); ?></strong> <code>[bcv_rates currency="USD" format="cards" show_date="yes" show_flags="yes" theme="modern"]</code></p>
                        <p><strong><?php _e('Documentación:', 'bcv-exchange-rates'); ?></strong> <a href="https://smartsites.es/tasa-oficial-del-bcv/" target="_blank">Ver Documentación</a></p>
                    </div>
                </div>
            </div>
        </div>
        <?php
    }
    
    /**
     * Página de configuración
     */
    public function settings_page() {
        ?>
        <div class="wrap">
            <h1><?php _e('Configuración de BCV Exchange Rates', 'bcv-exchange-rates'); ?></h1>
            
            <form method="post" action="options.php">
                <?php
                settings_fields('bcv_settings');
                do_settings_sections('bcv-settings');
                submit_button();
                ?>
            </form>
        </div>
        <?php
    }
    
    /**
     * Página de herramientas
     */
    public function tools_page() {
        ?>
        <div class="wrap">
            <h1><?php _e('Herramientas BCV Exchange Rates', 'bcv-exchange-rates'); ?></h1>
            
            <div class="bcv-tools-container">
                <div class="bcv-tool-section">
                    <h2><?php _e('Depuración', 'bcv-exchange-rates'); ?></h2>
                    <p><?php _e('Herramientas para diagnosticar problemas con el plugin.', 'bcv-exchange-rates'); ?></p>
                    
                    <button type="button" class="button" id="bcv-test-connection">
                        <?php _e('Probar Conexión', 'bcv-exchange-rates'); ?>
                    </button>
                    
                    <button type="button" class="button" id="bcv-view-logs">
                        <?php _e('Ver Logs', 'bcv-exchange-rates'); ?>
                    </button>
                </div>
                
                <div class="bcv-tool-section">
                    <h2><?php _e('Mantenimiento', 'bcv-exchange-rates'); ?></h2>
                    <p><?php _e('Herramientas de mantenimiento y limpieza.', 'bcv-exchange-rates'); ?></p>
                    
                    <button type="button" class="button button-warning" id="bcv-reset-plugin">
                        <?php _e('Restablecer Plugin', 'bcv-exchange-rates'); ?>
                    </button>
                </div>
            </div>
        </div>
        <?php
    }
    
    /**
     * Página de personalización
     */
    public function customizer_page() {
        if (!current_user_can('manage_options')) {
            wp_die(__('No tienes permisos suficientes para acceder a esta página.'));
        }
        
        // Cargar el personalizador
        $customizer = BCV_Customizer::get_instance();
        $settings = $customizer->get_settings();
        
        // Incluir vista del personalizador
        ?>
        <div class="wrap">
            <h1><?php _e('Personalizar Apariencia BCV Exchange Rates', 'bcv-exchange-rates'); ?></h1>
            
            <div class="bcv-customizer-container">
                <div class="bcv-customizer-sidebar">
                    <form method="post" action="options.php" id="bcv-customizer-form">
                        <?php
                        settings_fields('bcv_customizer_settings');
                        ?>
                        
                        <!-- Sección de Colores -->
                        <div class="bcv-customizer-section" data-section="colors">
                            <h2><?php _e('Colores', 'bcv-exchange-rates'); ?></h2>
                            
                            <div class="bcv-customizer-control">
                                <label for="bcv_color_primary"><?php _e('Color Principal', 'bcv-exchange-rates'); ?></label>
                                <input type="color" id="bcv_color_primary" name="bcv_customizer_settings[colors][primary]" 
                                       value="<?php echo esc_attr($settings['colors']['primary']); ?>" class="bcv-color-picker">
                                <p class="description"><?php _e('Color principal para títulos y elementos destacados', 'bcv-exchange-rates'); ?></p>
                            </div>
                            
                            <div class="bcv-customizer-control">
                                <label for="bcv_color_accent"><?php _e('Color de Acento', 'bcv-exchange-rates'); ?></label>
                                <input type="color" id="bcv_color_accent" name="bcv_customizer_settings[colors][accent]" 
                                       value="<?php echo esc_attr($settings['colors']['accent']); ?>" class="bcv-color-picker">
                                <p class="description"><?php _e('Color de acento para bordes y elementos decorativos', 'bcv-exchange-rates'); ?></p>
                            </div>
                            
                            <div class="bcv-customizer-control">
                                <label for="bcv_color_background"><?php _e('Color de Fondo', 'bcv-exchange-rates'); ?></label>
                                <input type="color" id="bcv_color_background" name="bcv_customizer_settings[colors][background]" 
                                       value="<?php echo esc_attr($settings['colors']['background']); ?>" class="bcv-color-picker">
                                <p class="description"><?php _e('Color de fondo del contenedor', 'bcv-exchange-rates'); ?></p>
                            </div>
                            
                            <div class="bcv-customizer-control">
                                <label for="bcv_color_surface"><?php _e('Color de Superficie', 'bcv-exchange-rates'); ?></label>
                                <input type="color" id="bcv_color_surface" name="bcv_customizer_settings[colors][surface]" 
                                       value="<?php echo esc_attr($settings['colors']['surface']); ?>" class="bcv-color-picker">
                                <p class="description"><?php _e('Color de fondo para tarjetas y elementos flotantes', 'bcv-exchange-rates'); ?></p>
                            </div>
                            
                            <div class="bcv-customizer-control">
                                <label for="bcv_color_text"><?php _e('Color de Texto', 'bcv-exchange-rates'); ?></label>
                                <input type="color" id="bcv_color_text" name="bcv_customizer_settings[colors][text]" 
                                       value="<?php echo esc_attr($settings['colors']['text']); ?>" class="bcv-color-picker">
                                <p class="description"><?php _e('Color principal del texto', 'bcv-exchange-rates'); ?></p>
                            </div>
                            
                            <div class="bcv-customizer-control">
                                <label for="bcv_color_muted"><?php _e('Color de Texto Secundario', 'bcv-exchange-rates'); ?></label>
                                <input type="color" id="bcv_color_muted" name="bcv_customizer_settings[colors][muted]" 
                                       value="<?php echo esc_attr($settings['colors']['muted']); ?>" class="bcv-color-picker">
                                <p class="description"><?php _e('Color para texto secundario y descripciones', 'bcv-exchange-rates'); ?></p>
                            </div>
                        </div>
                        
                        <!-- Sección de Tipografía -->
                        <div class="bcv-customizer-section" data-section="typography">
                            <h2><?php _e('Tipografía', 'bcv-exchange-rates'); ?></h2>
                            
                            <div class="bcv-customizer-control">
                                <label for="bcv_font_size_base"><?php _e('Tamaño de Fuente Base', 'bcv-exchange-rates'); ?></label>
                                <input type="range" id="bcv_font_size_base" name="bcv_customizer_settings[typography][font_size_base]" 
                                       min="12" max="20" value="<?php echo esc_attr($settings['typography']['font_size_base']); ?>" class="bcv-range-slider">
                                <span class="bcv-range-value"><?php echo esc_html($settings['typography']['font_size_base']); ?>px</span>
                                <p class="description"><?php _e('Tamaño base de fuente para el contenido', 'bcv-exchange-rates'); ?></p>
                            </div>
                            
                            <div class="bcv-customizer-control">
                                <label for="bcv_font_size_rates"><?php _e('Tamaño de Fuente de Precios', 'bcv-exchange-rates'); ?></label>
                                <input type="range" id="bcv_font_size_rates" name="bcv_customizer_settings[typography][font_size_rates]" 
                                       min="16" max="32" value="<?php echo esc_attr($settings['typography']['font_size_rates']); ?>" class="bcv-range-slider">
                                <span class="bcv-range-value"><?php echo esc_html($settings['typography']['font_size_rates']); ?>px</span>
                                <p class="description"><?php _e('Tamaño de fuente para los precios/tasas', 'bcv-exchange-rates'); ?></p>
                            </div>
                            
                            <div class="bcv-customizer-control">
                                <label for="bcv_font_weight_rates"><?php _e('Grosor de Fuente de Precios', 'bcv-exchange-rates'); ?></label>
                                <select id="bcv_font_weight_rates" name="bcv_customizer_settings[typography][font_weight_rates]">
                                    <option value="300" <?php selected($settings['typography']['font_weight_rates'], '300'); ?>>Ligero (300)</option>
                                    <option value="400" <?php selected($settings['typography']['font_weight_rates'], '400'); ?>>Normal (400)</option>
                                    <option value="500" <?php selected($settings['typography']['font_weight_rates'], '500'); ?>>Medio (500)</option>
                                    <option value="600" <?php selected($settings['typography']['font_weight_rates'], '600'); ?>>Semi-negrita (600)</option>
                                    <option value="700" <?php selected($settings['typography']['font_weight_rates'], '700'); ?>>Negrita (700)</option>
                                </select>
                                <p class="description"><?php _e('Grosor de la fuente para los precios', 'bcv-exchange-rates'); ?></p>
                            </div>
                            
                            <div class="bcv-customizer-control">
                                <label for="bcv_line_height"><?php _e('Altura de Línea', 'bcv-exchange-rates'); ?></label>
                                <input type="range" id="bcv_line_height" name="bcv_customizer_settings[typography][line_height]" 
                                       min="1" max="2" step="0.1" value="<?php echo esc_attr($settings['typography']['line_height']); ?>" class="bcv-range-slider">
                                <span class="bcv-range-value"><?php echo esc_html($settings['typography']['line_height']); ?></span>
                                <p class="description"><?php _e('Altura de línea para mejorar la legibilidad', 'bcv-exchange-rates'); ?></p>
                            </div>
                        </div>
                        
                        <!-- Sección de Espaciado -->
                        <div class="bcv-customizer-section" data-section="spacing">
                            <h2><?php _e('Espaciado y Márgenes', 'bcv-exchange-rates'); ?></h2>
                            
                            <div class="bcv-customizer-control">
                                <label for="bcv_margin_container"><?php _e('Margen del Contenedor', 'bcv-exchange-rates'); ?></label>
                                <input type="range" id="bcv_margin_container" name="bcv_customizer_settings[spacing][margin_container]" 
                                       min="0" max="50" value="<?php echo esc_attr($settings['spacing']['margin_container']); ?>" class="bcv-range-slider">
                                <span class="bcv-range-value"><?php echo esc_html($settings['spacing']['margin_container']); ?>px</span>
                                <p class="description"><?php _e('Margen exterior del contenedor principal', 'bcv-exchange-rates'); ?></p>
                            </div>
                            
                            <div class="bcv-customizer-control">
                                <label for="bcv_padding_container"><?php _e('Padding del Contenedor', 'bcv-exchange-rates'); ?></label>
                                <input type="range" id="bcv_padding_container" name="bcv_customizer_settings[spacing][padding_container]" 
                                       min="10" max="40" value="<?php echo esc_attr($settings['spacing']['padding_container']); ?>" class="bcv-range-slider">
                                <span class="bcv-range-value"><?php echo esc_html($settings['spacing']['padding_container']); ?>px</span>
                                <p class="description"><?php _e('Espaciado interno del contenedor', 'bcv-exchange-rates'); ?></p>
                            </div>
                            
                            <div class="bcv-customizer-control">
                                <label for="bcv_border_radius"><?php _e('Radio de Borde', 'bcv-exchange-rates'); ?></label>
                                <input type="range" id="bcv_border_radius" name="bcv_customizer_settings[spacing][border_radius]" 
                                       min="0" max="20" value="<?php echo esc_attr($settings['spacing']['border_radius']); ?>" class="bcv-range-slider">
                                <span class="bcv-range-value"><?php echo esc_html($settings['spacing']['border_radius']); ?>px</span>
                                <p class="description"><?php _e('Redondez de las esquinas de los elementos', 'bcv-exchange-rates'); ?></p>
                            </div>
                            
                            <div class="bcv-customizer-control">
                                <label for="bcv_shadow_intensity"><?php _e('Intensidad de Sombra', 'bcv-exchange-rates'); ?></label>
                                <input type="range" id="bcv_shadow_intensity" name="bcv_customizer_settings[spacing][shadow_intensity]" 
                                       min="0" max="50" value="<?php echo esc_attr($settings['spacing']['shadow_intensity']); ?>" class="bcv-range-slider">
                                <span class="bcv-range-value"><?php echo esc_html($settings['spacing']['shadow_intensity']); ?>%</span>
                                <p class="description"><?php _e('Intensidad de las sombras de los elementos', 'bcv-exchange-rates'); ?></p>
                            </div>
                        </div>
                        
                        <?php submit_button(); ?>
                    </form>
                </div>
                
                <!-- Vista previa -->
                <div class="bcv-customizer-preview">
                    <div class="bcv-preview-header">
                        <h2><?php _e('Vista Previa', 'bcv-exchange-rates'); ?></h2>
                        <button type="button" class="button button-secondary" id="bcv-refresh-preview">
                            <?php _e('Actualizar Vista Previa', 'bcv-exchange-rates'); ?>
                        </button>
                    </div>
                    <div class="bcv-preview-content">
                        <div class="bcv-preview-wrapper">
                            <div class="bcv-exchange-rates bcv-theme-custom">
                                <div class="bcv-last-updated">
                                    <span class="bcv-label">Actualizado:</span>
                                    <span class="bcv-date"><?php echo current_time('mysql'); ?></span>
                                </div>
                                <table class="bcv-rates-table">
                                    <thead>
                                        <tr>
                                            <th>Moneda</th>
                                            <th>Tasa BCV</th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        <tr>
                                            <td class="bcv-currency">
                                                <span class="bcv-flag flag-us"></span>
                                                <span class="bcv-code">USD</span>
                                                <span class="bcv-name">Dólar Americano</span>
                                            </td>
                                            <td class="bcv-rate">243,11 Bs</td>
                                        </tr>
                                        <tr>
                                            <td class="bcv-currency">
                                                <span class="bcv-flag flag-eu"></span>
                                                <span class="bcv-code">EUR</span>
                                                <span class="bcv-name">Euro</span>
                                            </td>
                                            <td class="bcv-rate">280,05 Bs</td>
                                        </tr>
                                        <tr>
                                            <td class="bcv-currency">
                                                <span class="bcv-flag flag-cn"></span>
                                                <span class="bcv-code">CNY</span>
                                                <span class="bcv-name">Yuan Chino</span>
                                            </td>
                                            <td class="bcv-rate">34,21 Bs</td>
                                        </tr>
                                    </tbody>
                                </table>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
        <?php
    }
    
    /**
     * Registrar configuraciones
     */
    public function register_settings() {
        register_setting('bcv_settings', 'bcv_options', array($this, 'sanitize_options'));
        
        add_settings_section(
            'bcv_general',
            __('Configuración General', 'bcv-exchange-rates'),
            array($this, 'general_section_callback'),
            'bcv-settings'
        );
        
        add_settings_field(
            'bcv_update_interval',
            __('Intervalo de Actualización', 'bcv-exchange-rates'),
            array($this, 'update_interval_callback'),
            'bcv-settings',
            'bcv_general'
        );
        
        add_settings_field(
            'bcv_default_format',
            __('Formato por Defecto', 'bcv-exchange-rates'),
            array($this, 'default_format_callback'),
            'bcv-settings',
            'bcv_general'
        );
        
        add_settings_field(
            'bcv_default_theme',
            __('Tema por Defecto', 'bcv-exchange-rates'),
            array($this, 'default_theme_callback'),
            'bcv-settings',
            'bcv_general'
        );
    }
    
    /**
     * Callbacks de configuración
     */
    public function general_section_callback() {
        echo '<p>' . __('Configura las opciones generales del plugin.', 'bcv-exchange-rates') . '</p>';
    }
    
    public function update_interval_callback() {
        $options = get_option('bcv_options');
        $interval = isset($options['update_interval']) ? $options['update_interval'] : HOUR_IN_SECONDS;
        ?>
        <select name="bcv_options[update_interval]">
            <option value="900" <?php selected($interval, 900); ?>>15 minutos</option>
            <option value="1800" <?php selected($interval, 1800); ?>>30 minutos</option>
            <option value="3600" <?php selected($interval, 3600); ?>>1 hora</option>
            <option value="7200" <?php selected($interval, 7200); ?>>2 horas</option>
            <option value="14400" <?php selected($interval, 14400); ?>>4 horas</option>
        </select>
        <p class="description"><?php _e('Frecuencia de actualización de los tipos de cambio.', 'bcv-exchange-rates'); ?></p>
        <?php
    }
    
    public function default_format_callback() {
        $options = get_option('bcv_options');
        $format = isset($options['default_format']) ? $options['default_format'] : 'table';
        ?>
        <select name="bcv_options[default_format]">
            <option value="table" <?php selected($format, 'table'); ?>>Tabla</option>
            <option value="cards" <?php selected($format, 'cards'); ?>>Tarjetas</option>
            <option value="list" <?php selected($format, 'list'); ?>>Lista</option>
        </select>
        <p class="description"><?php _e('Formato por defecto para el shortcode.', 'bcv-exchange-rates'); ?></p>
        <?php
    }
    
    public function default_theme_callback() {
        $options = get_option('bcv_options');
        $theme = isset($options['default_theme']) ? $options['default_theme'] : 'modern';
        ?>
        <select name="bcv_options[default_theme]">
            <option value="modern" <?php selected($theme, 'modern'); ?>>Moderno</option>
            <option value="classic" <?php selected($theme, 'classic'); ?>>Clásico</option>
            <option value="minimal" <?php selected($theme, 'minimal'); ?>>Minimalista</option>
        </select>
        <p class="description"><?php _e('Tema por defecto para la visualización.', 'bcv-exchange-rates'); ?></p>
        <?php
    }
    
    /**
     * Sanitizar opciones
     */
    public function sanitize_options($input) {
        $sanitized = array();
        
        if (isset($input['update_interval'])) {
            $sanitized['update_interval'] = absint($input['update_interval']);
        }
        
        if (isset($input['default_format'])) {
            $sanitized['default_format'] = sanitize_text_field($input['default_format']);
        }
        
        if (isset($input['default_theme'])) {
            $sanitized['default_theme'] = sanitize_text_field($input['default_theme']);
        }
        
        return $sanitized;
    }
    
    /**
     * Agregar widget al dashboard
     */
    public function add_dashboard_widget() {
        wp_add_dashboard_widget(
            'bcv_exchange_rates_widget',
            __('Tipos de Cambio BCV', 'bcv-exchange-rates'),
            array($this, 'dashboard_widget_content')
        );
    }
    
    /**
     * Contenido del widget del dashboard
     */
    public function dashboard_widget_content() {
        $data = get_transient('bcv_exchange_rates_data');
        
        if ($data && !empty($data['rates'])) {
            ?>
            <div class="bcv-dashboard-widget">
                <?php foreach ($data['rates'] as $currency => $rate): ?>
                    <div class="bcv-rate-item">
                        <span class="bcv-currency"><?php echo esc_html($currency); ?></span>
                        <span class="bcv-rate"><?php echo number_format($rate, 2, ',', '.'); ?> Bs</span>
                    </div>
                <?php endforeach; ?>
                <p class="bcv-last-updated">
                    <?php _e('Actualizado:', 'bcv-exchange-rates'); ?> <?php echo esc_html($data['last_updated']); ?>
                </p>
            </div>
            <?php
        } else {
            echo '<p>' . __('No hay datos disponibles.', 'bcv-exchange-rates') . '</p>';
        }
    }
    
    /**
     * Scripts y estilos del administrador
     */
    public function enqueue_admin_scripts($hook) {
        if (strpos($hook, 'bcv') === false) {
            return;
        }
        
        wp_enqueue_style(
            'bcv-admin-style',
            plugins_url('assets/css/admin.css', dirname(__FILE__)),
            array(),
            '1.0.0'
        );
        
        // Enqueue customizer styles if on customizer page
        if ($hook === 'bcv-exchange-rates_page_bcv-customizer') {
            wp_enqueue_style(
                'bcv-customizer-style',
                plugins_url('assets/css/customizer.css', dirname(__FILE__)),
                array('wp-color-picker'),
                '1.0.0'
            );
            
            wp_enqueue_script(
                'bcv-customizer',
                plugins_url('assets/js/customizer.js', dirname(__FILE__)),
                array('jquery', 'wp-color-picker'),
                '1.0.0',
                true
            );
            
            wp_localize_script('bcv-customizer', 'bcvCustomizer', array(
                'ajax_url' => admin_url('admin-ajax.php'),
                'nonce' => wp_create_nonce('bcv_customizer_nonce'),
            ));
        }
        
        wp_enqueue_script(
            'bcv-admin-script',
            plugins_url('assets/js/admin.js', dirname(__FILE__)),
            array('jquery'),
            '1.0.0',
            true
        );
        
        wp_localize_script('bcv-admin-script', 'bcvAdmin', array(
            'ajax_url' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('bcv_admin_nonce'),
        ));
    }
    
    /**
     * Notificaciones de administración
     */
    public function admin_notices() {
        $screen = get_current_screen();
        
        if ($screen->id !== 'toplevel_page_bcv-exchange-rates') {
            return;
        }
        
        $status = bcv_get_service_status();
        
        if ($status['status'] === 'error') {
            ?>
            <div class="notice notice-error">
                <p><strong><?php _e('BCV Exchange Rates:', 'bcv-exchange-rates'); ?></strong> 
                <?php echo esc_html($status['message']); ?></p>
            </div>
            <?php
        }
    }
}

// Inicializar administración
BCV_Admin::get_instance();

// Funciones AJAX para el administrador
add_action('wp_ajax_bcv_force_update', 'bcv_ajax_force_update');
add_action('wp_ajax_bcv_clear_cache', 'bcv_ajax_clear_cache');
add_action('wp_ajax_bcv_test_connection', 'bcv_ajax_test_connection');

function bcv_ajax_force_update() {
    check_ajax_referer('bcv_admin_nonce', 'nonce');
    
    if (!current_user_can('manage_options')) {
        wp_die(__('No tienes permisos suficientes.'));
    }
    
    $result = bcv_force_update_rates();
    
    wp_send_json($result);
}

function bcv_ajax_clear_cache() {
    check_ajax_referer('bcv_admin_nonce', 'nonce');
    
    if (!current_user_can('manage_options')) {
        wp_die(__('No tienes permisos suficientes.'));
    }
    
    delete_transient('bcv_exchange_rates_data');
    delete_transient('bcv_exchange_rates_backup');
    
    wp_send_json(array(
        'success' => true,
        'message' => __('Caché limpiado correctamente.')
    ));
}

function bcv_ajax_test_connection() {
    check_ajax_referer('bcv_admin_nonce', 'nonce');
    
    if (!current_user_can('manage_options')) {
        wp_die(__('No tienes permisos suficientes.'));
    }
    
    $updater = BCV_Updater::get_instance();
    $test_data = $updater->fetch_rates();
    
    $success = !empty($test_data['rates']) && $updater->validate_rates($test_data['rates']);
    
    wp_send_json(array(
        'success' => $success,
        'message' => $success ? __('Conexión exitosa.') : __('Error de conexión.'),
        'data' => $test_data
    ));
}