<?php
/**
 * BCV Exchange Rates Updater
 * Sistema de actualización de tipos de cambio
 */

if (!defined('ABSPATH')) {
    exit;
}

class BCV_Updater {
    
    private static $instance = null;
    private $api_url = 'https://www.bcv.org.ve/';
    private $user_agent = 'BCV Exchange Rates Plugin/1.0.0';
    
    public static function get_instance() {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    private function __construct() {
        // Constructor privado para singleton
    }
    
    /**
     * Obtener tipos de cambio del BCV
     */
    public function fetch_rates() {
        $args = array(
            'timeout' => 30,
            'user-agent' => $this->user_agent,
            'headers' => array(
                'Accept' => 'text/html,application/xhtml+xml,application/xml;q=0.9,*/*;q=0.8',
                'Accept-Language' => 'es-VE,es;q=0.9,en;q=0.8',
                'Cache-Control' => 'no-cache',
            ),
            'sslverify' => false // Mantiene la corrección temporal para cURL error 60
        );
        
        $response = wp_remote_get($this->api_url, $args);
        
        if (is_wp_error($response)) {
            return $this->handle_error($response);
        }
        
        $body = wp_remote_retrieve_body($response);
        $http_code = wp_remote_retrieve_response_code($response);
        
        if ($http_code !== 200) {
            return $this->handle_error(new WP_Error('http_error', 'HTTP Error: ' . $http_code));
        }
        
        return $this->parse_exchange_rates($body);
    }
    
    /**
     * Parsear HTML para extraer tipos de cambio
     */
    private function parse_exchange_rates($html) {
        $rates = array();
        
        // Mapeo de códigos de moneda a IDs de DIV en la página del BCV.
        // Si el sitio del BCV cambia, estas IDs deben actualizarse.
        $currency_selectors = array(
            'USD' => 'dolar',
            'EUR' => 'euro',
            'CNY' => 'yuan',
            'TRY' => 'lira', 
            'RUB' => 'rublo',
        );
        
        // Usamos la clase nativa DOMDocument para parsear HTML
        libxml_use_internal_errors(true); // Suprimir errores de HTML mal formado
        $dom = new DOMDocument();
        // Aseguramos el charset UTF-8
        $dom->loadHTML(mb_convert_encoding($html, 'HTML-ENTITIES', 'UTF-8'));
        $xpath = new DOMXPath($dom);
        libxml_use_internal_errors(false);
        
        // Iterar sobre todas las monedas
        foreach ($currency_selectors as $code => $id) {
            
            // Selector XPath: Busca el primer <strong> dentro del div con el ID de la moneda
            // Esto captura la Tasa de Venta oficial.
            $rate_node = $xpath->query("//div[@id='{$id}']//strong[1]"); 
            
            if ($rate_node->length > 0) {
                $raw_rate = $rate_node->item(0)->textContent;
                
                // Limpiar y formatear la tasa:
                // 1. Eliminar todo lo que no sea dígito, coma o punto.
                $cleaned_rate = preg_replace('/[^0-9,.]/', '', $raw_rate);
                
                // 2. Reemplazar la coma (,) por punto (.) para que PHP lo trate como float.
                $final_rate = floatval(str_replace(',', '.', $cleaned_rate));
                
                if ($final_rate > 0) {
                    $rates[$code] = $final_rate; // Almacenar la tasa encontrada
                }
            }
        }
        
        // Determinar el estado
        $status = empty($rates) ? 'error' : 'success';
        $error_message = empty($rates) ? 'No se pudo extraer ninguna tasa. El selector XPath puede haber cambiado.' : '';

        return array(
            'rates' => $rates,
            'last_updated' => current_time('mysql'),
            'source' => 'BCV',
            'status' => $status,
            'error_message' => $error_message
        );
    }
    
    /**
     * Manejar errores de actualización
     */
    private function handle_error($error) {
        $error_message = is_wp_error($error) ? $error->get_error_message() : 'Unknown error';
        
        // Log del error
        error_log('BCV Exchange Rates Error: ' . $error_message);
        
        // Intentar usar datos anteriores si están disponibles
        $cached_data = get_transient('bcv_exchange_rates_backup');
        
        if ($cached_data !== false) {
            $cached_data['status'] = 'error';
            $cached_data['error_message'] = $error_message;
            return $cached_data;
        }
        
        // Si no hay datos anteriores, usar datos de ejemplo (último recurso)
        return array(
            'rates' => array(
                'USD' => 243.11050000,
                'EUR' => 280.04870937,
                'CNY' => 34.20574620,
                'TRY' => 5.72783475,
                'RUB' => 3.07773768,
            ),
            'last_updated' => current_time('mysql'),
            'source' => 'BCV',
            'status' => 'error',
            'error_message' => $error_message
        );
    }
    
    /**
     * Validar datos de tipos de cambio
     */
    public function validate_rates($rates) {
        if (!is_array($rates) || empty($rates)) {
            return false;
        }
        
        foreach ($rates as $currency => $rate) {
            if (!is_string($currency) || strlen($currency) !== 3) {
                return false;
            }
            
            if (!is_numeric($rate) || $rate <= 0) {
                return false;
            }
        }
        
        return true;
    }
    
    /**
     * Obtener fecha de última actualización
     */
    public function get_last_update_time() {
        $data = get_transient('bcv_exchange_rates_data');
        
        if ($data && isset($data['last_updated'])) {
            return $data['last_updated'];
        }
        
        return 'Nunca';
    }
    
    /**
     * Obtener estado del servicio
     */
    public function get_service_status() {
        $data = get_transient('bcv_exchange_rates_data');
        
        if (!$data) {
            return array(
                'status' => 'error',
                'message' => 'No hay datos disponibles',
                'last_update' => 'Nunca'
            );
        }
        
        $status = isset($data['status']) ? $data['status'] : 'success';
        $message = isset($data['error_message']) ? $data['error_message'] : 'Servicio funcionando correctamente';
        
        return array(
            'status' => $status,
            'message' => $message,
            'last_update' => $data['last_updated'],
            'next_update' => $this->get_next_update_time()
        );
    }
    
    /**
     * Obtener tiempo hasta próxima actualización
     */
    private function get_next_update_time() {
        $cron_schedules = wp_get_schedules();
        $next_cron = wp_next_scheduled('bcv_update_rates_event');
        
        if ($next_cron) {
            return date('Y-m-d H:i:s', $next_cron);
        }
        
        return 'No programado';
    }
    
    /**
     * Forzar actualización manual
     */
    public function force_update() {
        $new_data = $this->fetch_rates();
        
        if ($this->validate_rates($new_data['rates'])) {
            // Guardar como datos principales
            set_transient('bcv_exchange_rates_data', $new_data, HOUR_IN_SECONDS);
            
            // Guardar como respaldo
            set_transient('bcv_exchange_rates_backup', $new_data, DAY_IN_SECONDS);
            
            return array(
                'success' => true,
                'message' => 'Datos actualizados correctamente',
                'rates' => $new_data['rates']
            );
        }
        
        return array(
            'success' => false,
            'message' => 'Error al validar los datos obtenidos',
            'rates' => array()
        );
    }
}

// Función auxiliar para obtener el estado del servicio
function bcv_get_service_status() {
    $updater = BCV_Updater::get_instance();
    return $updater->get_service_status();
}

// Función auxiliar para forzar actualización
function bcv_force_update_rates() {
    $updater = BCV_Updater::get_instance();
    return $updater->force_update();
}