<?php
/**
 * Plugin Name: Description AI Product for WooCommerce
 * Plugin URI:  https://smartsites.es/desc-ai-product/
 * Description: Automatically generate product descriptions by analyzing your images with artificial intelligence.
 * Version:     1.0.2
 * Author:      SmartSites
 * Author URI:  https://smartsites.es/
 * License:     GPL-2.0+
 * Text Domain: desc-ai-product
 * Domain Path: /languages
 * Tested up to: 6.9
 */



// Si el archivo se llama directamente, abortar.
if (!defined('ABSPATH')) {
    exit;
}

// 1. Añadir Página de Configuración bajo el menú de WooCommerce
add_action('admin_menu', 'wc_ai_add_submenu_page', 99);
function wc_ai_add_submenu_page()
{
    add_submenu_page(
        'woocommerce',
        __('IA OpenRouter', 'desc-ai-product'),
        __('IA OpenRouter', 'desc-ai-product'),
        'manage_options',
        'wc-ai-settings',
        'wc_ai_settings_page_html'
    );
}

// HTML de la Página de Configuración
function wc_ai_settings_page_html()
{
    if (!current_user_can('manage_options')) {
        return;
    }

    //Obtener los valores actuales de la base de datos (evita el error de variable indefinida)
    $api_key = get_option('wc_ai_openrouter_api_key', '');
    $selected_model = get_option('wc_ai_openrouter_model', 'openrouter/auto');


    if (isset($_POST['wc_ai_api_key']) && check_admin_referer('wc_ai_save_settings', 'wc_ai_nonce')) {
        update_option('wc_ai_openrouter_api_key', sanitize_text_field(wp_unslash($_POST['wc_ai_api_key'])));

        $api_key = get_option('wc_ai_openrouter_api_key', '');

        echo '<div class="notice notice-success is-dismissible"><p>' . esc_html__('Ajustes guardados correctamente.', 'desc-ai-product') . '</p></div>';
    }

?>
    <div class="wrap">
        <h1><?php esc_html_e('Configuración de OpenRouter para WooCommerce', 'desc-ai-product'); ?></h1>
        <p><?php esc_html_e('Configura tu clave de API y elige un modelo de Inteligencia Artificial gratuito con soporte de visión.', 'desc-ai-product'); ?></p>
        <form action="" method="post">
            <?php wp_nonce_field('wc_ai_save_settings', 'wc_ai_nonce'); ?>
            <table class="form-table">
                <tr>
                    <th scope="row"><label for="wc_ai_api_key"><?php esc_html_e('OpenRouter API Key', 'desc-ai-product'); ?></label></th>
                    <td>
                        <input type="password" name="wc_ai_api_key" id="wc_ai_api_key" value="<?php echo esc_attr($api_key); ?>" class="regular-text">
                        <p class="description">
                            <?php
    printf(
        wp_kses(
        /* translators: %s: URL to OpenRouter keys page */
        __('Puedes obtener tu API clave en <a href="%s" target="_blank">OpenRouter</a>.', 'desc-ai-product'),
        array('a' => array('href' => array(), 'target' => array()))
    ),
        'https://openrouter.ai/keys'
    );
?>
                        </p>
                    </td>
                </tr>
            </table>
            <?php submit_button(__('Guardar Cambios', 'desc-ai-product')); ?>
        </form>
    </div>
    <?php
}

// 2. Añadir el botón en el metabox "Datos del producto" de WooCommerce
add_action('woocommerce_product_options_general_product_data', 'wc_ai_add_generate_button');
function wc_ai_add_generate_button()
{
    echo '<div class="options_group">';
    echo '<p class="form-field">';
    echo '<label>' . esc_html__('Generador con IA', 'desc-ai-product') . '</label>';
    echo '<button type="button" id="wc-ai-generate-btn" class="button button-primary">' . esc_html__('Generar Descripción con IA', 'desc-ai-product') . '</button>';
    echo '<span id="wc-ai-status" style="margin-left: 10px; font-weight: 500;"></span>';
    echo '<span class="description" style="margin-top: 5px; display: block; clear: both;">' . esc_html__('Utiliza la imagen destacada actual para redactar la descripción. (Guarda el producto primero si acabas de cambiar la imagen).', 'desc-ai-product') . '</span>';
    echo '</p>';
    echo '</div>';
}

// 3. Enqueue de Scripts para manejar el botón y la inserción del texto
add_action('admin_enqueue_scripts', 'wc_ai_enqueue_scripts');
function wc_ai_enqueue_scripts($hook)
{
    global $post;

    // Solo cargar en la pantalla de edición de productos
    if (($hook === 'post-new.php' || $hook === 'post.php') && isset($post) && 'product' === $post->post_type) {
        wp_enqueue_script('jquery');

        $nonce = wp_create_nonce('wc_ai_generate_nonce');

        $btn_text = esc_js(__('Generar Descripción con IA', 'desc-ai-product'));
        $btn_generating = esc_js(__('Generando...', 'desc-ai-product'));
        $status_analyzing = esc_js(__('Analizando imagen con IA...', 'desc-ai-product'));
        $status_success = esc_js(__('¡Descripción generada y añadida al editor!', 'desc-ai-product'));
        $status_error_suffix = esc_js(__('Error: ', 'desc-ai-product'));
        $status_conn_error = esc_js(__('Error de conexión con el servidor.', 'desc-ai-product'));

        $script = "
        jQuery(document).ready(function($) {
            $('#wc-ai-generate-btn').on('click', function(e) {
                e.preventDefault();
                
                var btn = $(this);
                var status = $('#wc-ai-status');
                
                btn.prop('disabled', true).text('{$btn_generating}');
                status.html('<span class=\"spinner is-active\" style=\"float:none; margin:0 5px 0 0;\"></span> ' + '{$status_analyzing}').show();
                
                $.ajax({
                    url: ajaxurl,
                    type: 'POST',
                    data: {
                        action: 'wc_ai_generate_description',
                        post_id: $('#post_ID').val(),
                        nonce: '{$nonce}'
                    },
                    success: function(response) {
                        btn.prop('disabled', false).text('{$btn_text}');
                        if (response.success) {
                            status.html('<span style=\"color:#2271b1;\">{$status_success}</span>');
                            var generatedText = response.data;
                            
                            // 1. Intentar insertar en Gutenberg (Block Editor)
                            if (typeof wp !== 'undefined' && wp.data && wp.data.dispatch('core/editor')) {
                                var currentContent = wp.data.select('core/editor').getEditedPostAttribute('content') || '';
                                var newContent = currentContent + (currentContent ? '\\n\\n' : '') + generatedText;
                                wp.data.dispatch('core/editor').editPost({ content: newContent });
                            } 
                            // 2. Intentar insertar en TinyMCE (Visual Editor Clásico)
                            else if (typeof tinyMCE !== 'undefined' && tinyMCE.get('content') && !tinyMCE.get('content').isHidden()) {
                                tinyMCE.get('content').execCommand('mceInsertContent', false, generatedText);
                            } 
                            // 3. Fallback a Editor de Texto (HTML)
                            else if ($('#content').length) {
                                var val = $('#content').val();
                                $('#content').val(val + (val ? '\\n\\n' : '') + generatedText);
                            }
                            
                            setTimeout(function(){ status.fadeOut(); }, 5000);
                        } else {
                            status.html('<span style=\"color:#d63638;\">{$status_error_suffix}' + response.data + '</span>');
                        }
                    },
                    error: function() {
                        btn.prop('disabled', false).text('{$btn_text}');
                        status.html('<span style=\"color:#d63638;\">{$status_conn_error}</span>');
                    }
                });
            });
        });
        ";
        wp_add_inline_script('jquery', $script);
    }
}

// 4. Controlador AJAX para comunicarse con OpenRouter
add_action('wp_ajax_wc_ai_generate_description', 'wc_ai_generate_description_ajax');
function wc_ai_generate_description_ajax()
{
    // Verificación de Nonce
    check_ajax_referer('wc_ai_generate_nonce', 'nonce');

    // Verificación de Permisos (Solo editores)
    if (!current_user_can('edit_products')) {
        wp_send_json_error(__('No tienes permisos suficientes para realizar esta acción.', 'desc-ai-product'));
    }

    // Obtener y sanitizar el ID del producto
    $post_id = isset($_POST['post_id']) ? intval($_POST['post_id']) : 0;
    if (!$post_id) {
        wp_send_json_error(__('ID de producto no válido.', 'desc-ai-product'));
    }

    // Obtener la imagen destacada
    $thumbnail_id = get_post_thumbnail_id($post_id);
    if (!$thumbnail_id) {
        wp_send_json_error(__('El producto no tiene una imagen destacada guardada.', 'desc-ai-product'));
    }

    $image_url = wp_get_attachment_image_url($thumbnail_id, 'full');
    if (!$image_url) {
        wp_send_json_error(__('No se pudo obtener la URL de la imagen destacada.', 'desc-ai-product'));
    }

    // Obtener ruta local de la imagen para enviarla en base64
    $image_path = get_attached_file($thumbnail_id);
    if ($image_path && file_exists($image_path)) {
        $mime_type = mime_content_type($image_path);
        $image_data = base64_encode(file_get_contents($image_path));
        $image_payload = array(
            'url' => 'data:' . $mime_type . ';base64,' . $image_data
        );
    }
    else {
        // Fallback a URL (no funcionará en localhost con OpenRouter, pero sí en producción)
        $image_payload = array(
            'url' => $image_url
        );
    }

    // Obtener API Key y Modelo
    $api_key = get_option('wc_ai_openrouter_api_key', '');
    if (empty($api_key)) {
        wp_send_json_error(__('La clave de API de OpenRouter no está configurada.', 'desc-ai-product'));
    }

    // Forzar el modelo a openrouter/auto
    $selected_model = 'openrouter/auto';

    // Preparar la petición a OpenRouter (Obteniendo el idioma actual de forma segura)
    $current_locale = get_locale();
    $current_language = function_exists('locale_get_display_name') ? locale_get_display_name($current_locale, $current_locale) : $current_locale;

    $prompt = sprintf(
        /* translators: %s: nombre del idioma activo */
        __('Analiza esta imagen de producto y redacta una descripción de venta profesional, resaltando sus características y beneficios principales. El texto debe estar en el idioma de este código (locale): %s. Utiliza un tono persuasivo. IMPORTANTÍSIMO: Devuelve tu respuesta usando formato de texto o Markdown simple (para negritas y listas), pero NO envuelvas toda tu respuesta dentro de un bloque de código markdown (no uses ```).', 'desc-ai-product'),
        $current_language
    );

    $body = array(
        'model' => $selected_model,
        'messages' => array(
                array(
                'role' => 'user',
                'content' => array(
                        array(
                        'type' => 'text',
                        'text' => $prompt
                    ),
                        array(
                        'type' => 'image_url',
                        'image_url' => $image_payload
                    )
                )
            )
        )
    );

    // Enviar petición usando wp_remote_post
    $response = wp_remote_post('https://openrouter.ai/api/v1/chat/completions', array(
        'headers' => array(
            'Authorization' => 'Bearer ' . $api_key,
            'Content-Type' => 'application/json',
            'HTTP-Referer' => home_url(),
            'X-Title' => get_bloginfo('name')
        ),
        'body' => wp_json_encode($body),
        'timeout' => 45 // La IA puede tardar un poco en analizar la imagen
    ));

    // Manejo de errores de conexión de red
    if (is_wp_error($response)) {
        wp_send_json_error(__('Error de conexión: ', 'desc-ai-product') . $response->get_error_message());
    }

    $status_code = wp_remote_retrieve_response_code($response);
    $response_body = wp_remote_retrieve_body($response);
    $data = json_decode($response_body, true);

    // Manejo de errores de API
    if ($status_code !== 200) {
        $error_msg = isset($data['error']['message']) ? $data['error']['message'] : sprintf(
            /* translators: %d: HTTP response status code */
            __('Error en la respuesta de la API (Código: %d).', 'desc-ai-product'),
            $status_code
        );
        wp_send_json_error($error_msg);
    }

    // Inserción de la respuesta
    if (isset($data['choices'][0]['message']['content'])) {
        $generated_text = $data['choices'][0]['message']['content'];

        // Convertir formato básico Markdown a HTML para correcta visualización en el editor visual
        $html = '';
        $lines = explode("\n", $generated_text);
        $in_list = false;

        foreach ($lines as $line) {
            $line = trim($line);

            // Reemplazar Negritas de Markdown
            $line = preg_replace('/\*\*(.*?)\*\*/', '<strong>$1</strong>', $line);

            // Reemplazar Títulos de Markdown
            $line = preg_replace('/^\#\#\# (.*)/', '<h3>$1</h3>', $line);
            $line = preg_replace('/^\#\# (.*)/', '<h2>$1</h2>', $line);
            $line = preg_replace('/^\# (.*)/', '<h1>$1</h1>', $line);

            // Detectar Listas
            if (preg_match('/^[\*\-] (.*)/', $line, $match)) {
                if (!$in_list) {
                    $html .= "<ul>\n";
                    $in_list = true;
                }
                $html .= "<li>" . $match[1] . "</li>\n";
            }
            else {
                if ($in_list) {
                    $html .= "</ul>\n";
                    $in_list = false;
                }
                if (!empty($line) && !preg_match('/^<h[1-3]>/', $line)) {
                    $html .= "<p>$line</p>\n";
                }
                elseif (!empty($line)) {
                    $html .= "$line\n";
                }
            }
        }

        if ($in_list) {
            $html .= "</ul>\n";
        }

        wp_send_json_success($html);
    }
    else {
        wp_send_json_error(__('No se recibió texto de la IA.', 'desc-ai-product'));
    }
}
